// Photoshop Crop & Export Script (Final Updated Version)
// Processes all images in a selected source folder, crops them centrally to match
// the chosen size, converts to sRGB, and saves them to the output folder.

// ------------------------------------------------------
// CROP SIZE DEFINITIONS
// ------------------------------------------------------
var cropSizes = {
    "Standard": [
        {name: "5x3.5", width: 5, height: 3.5},
        {name: "5x4", width: 5, height: 4},
        {name: "6x4", width: 6, height: 4},
        {name: "7x5", width: 7, height: 5},
        {name: "7.5x5", width: 7.5, height: 5},
        {name: "8x6", width: 8, height: 6},
        {name: "9x6", width: 9, height: 6},
        {name: "10x7", width: 10, height: 7},
        {name: "10x8", width: 10, height: 8},
        {name: "12x8", width: 12, height: 8},
        {name: "12x9", width: 12, height: 9},
        {name: "12x10", width: 12, height: 10},
        {name: "14x10", width: 14, height: 10},
        {name: "14x11", width: 14, height: 11},
        {name: "15x10", width: 15, height: 10},
        {name: "16x12", width: 16, height: 12},
        {name: "18x12", width: 18, height: 12}
    ],
    "A Sizes": [
        {name: "A5", width: 8.27, height: 5.83},
        {name: "A4", width: 11.69, height: 8.27},
        {name: "A3", width: 16.54, height: 11.69}
    ],
    "Square": [
        {name: "5x5", width: 5, height: 5},
        {name: "6x6", width: 6, height: 6},
        {name: "8x8", width: 8, height: 8},
        {name: "10x10", width: 10, height: 10},
        {name: "11x11", width: 11, height: 11},
        {name: "12x12", width: 12, height: 12}
    ],
    "Panoramic": [
        {name: "10x5", width: 10, height: 5},
        {name: "12x5", width: 12, height: 5},
        {name: "14x8", width: 14, height: 8},
        {name: "15x5", width: 15, height: 5},
        {name: "15x6", width: 15, height: 6},
        {name: "17x6", width: 17, height: 6},
        {name: "18x8", width: 18, height: 8},
        {name: "20x8", width: 20, height: 8}
    ]
};

// ------------------------------------------------------
// MAIN
// ------------------------------------------------------
function main() {

    // --- Step 1: Ask user to select source folder ---
alert("Please select the folder containing the images you want to process.");
var sourceFolder = Folder.selectDialog("Select the folder containing images to crop:");
if (!sourceFolder) return;

// --- Step 2: Ask user to select or create output folder ---
alert("Now, please select (or create) the folder where the processed images will be saved.");
var outputFolder = Folder.selectDialog("Select the folder to save processed images:");
if (!outputFolder) return;


    // Show Size Selection Dialog
    var selectedSize = showCropDialog();
    if (selectedSize == null) return;

    // Get list of images
    var files = sourceFolder.getFiles(/\.(jpg|jpeg|tif|tiff|png|psd)$/i);

    if (files.length === 0) {
        alert("No compatible images found in the selected folder.");
        return;
    }

    // Set ruler units
    var originalUnits = app.preferences.rulerUnits;
    app.preferences.rulerUnits = Units.INCHES;

    // Process images
    for (var i = 0; i < files.length; i++) {
        try {
            var file = files[i];
            var doc = app.open(file);

            // Convert to 300 DPI
            doc.resizeImage(null, null, 300, ResampleMethod.BICUBIC);

            // Crop with orientation awareness
            cropImage(doc, selectedSize.width, selectedSize.height);

            // Convert to sRGB
            convertToSRGB(doc);

            // Save output file
            saveAsJPG(doc, selectedSize.name, outputFolder);

            doc.close(SaveOptions.DONOTSAVECHANGES);

        } catch (e) {
            alert("Error with file: " + file.name + "\n" + e.message);
        }
    }

    app.preferences.rulerUnits = originalUnits;

    alert("All images processed successfully.\nSaved in: " + outputFolder.fsName);
}

// ------------------------------------------------------
// CROP SIZE DIALOG WITH NOTICE
// ------------------------------------------------------
function showCropDialog() {
    var dialog = new Window("dialog", "Select Crop Size");
    dialog.orientation = "column";
    dialog.alignChildren = "fill";
    dialog.preferredSize.width = 400;


    // Category
    var group = dialog.add("group");
    group.add("statictext", undefined, "Category:");
    var catDrop = group.add("dropdownlist", undefined, []);

    for (var cat in cropSizes) {
        if (cropSizes.hasOwnProperty(cat)) {
            catDrop.add("item", cat);
        }
    }
    catDrop.selection = 0;

    // Size list
    var group2 = dialog.add("group");
    group2.add("statictext", undefined, "Size:");
    var sizeDrop = group2.add("dropdownlist", undefined, []);

    function inchesToMM(x) { return Math.round(x * 25.4); }

    function updateSizes() {
        sizeDrop.removeAll();
        var list = cropSizes[catDrop.selection.text];

        for (var i = 0; i < list.length; i++) {
            var w = list[i].width;
            var h = list[i].height;

            sizeDrop.add("item",
                list[i].name + "   (" +
                inchesToMM(w) + " x " + inchesToMM(h) + " mm)"
            );
        }

        sizeDrop.selection = 0;
    }

    catDrop.onChange = updateSizes;
    updateSizes();

    // Notice text
    // --- Notice box (multiline, readable, resizable) ---
    var notice = dialog.add(
    "edittext",
    undefined,
    "Instructions:\n" +
    "Select your print catagory, then choose your crop size.\n\n" +
    "Please Note: If your image apect ratio is diffrent from the chosen crop size image cropping will occur.",
    {multiline: true, readonly: true}
    );

// Set visible size of the notice box
notice.preferredSize.width = 350;
notice.preferredSize.height = 110;


    // Buttons
    var buttons = dialog.add("group");
    buttons.add("button", undefined, "OK");
    buttons.add("button", undefined, "Cancel");

    if (dialog.show() !== 1) return null;

    return cropSizes[catDrop.selection.text][sizeDrop.selection.index];
}

// ------------------------------------------------------
// ORIENTATION-AWARE CENTRAL CROP
// ------------------------------------------------------
function cropImage(doc, width, height) {

    var docW = doc.width.as("in");
    var docH = doc.height.as("in");

    var docIsLandscape = docW >= docH;
    var cropIsLandscape = width >= height;

    // Rotate crop orientation to match image
    if (docIsLandscape !== cropIsLandscape) {
        var t = width;
        width = height;
        height = t;
    }

    var targetRatio = width / height;
    var currentRatio = docW / docH;

    var left, top, right, bottom;

    if (currentRatio > targetRatio) {
        var cropWidth = docH * targetRatio;
        left = (docW - cropWidth) / 2;
        right = left + cropWidth;
        top = 0;
        bottom = docH;
    } else {
        var cropHeight = docW / targetRatio;
        top = (docH - cropHeight) / 2;
        bottom = top + cropHeight;
        left = 0;
        right = docW;
    }

    doc.crop([
        new UnitValue(left, "in"),
        new UnitValue(top, "in"),
        new UnitValue(right, "in"),
        new UnitValue(bottom, "in")
    ]);

    // Resize to final dimensions
    doc.resizeImage(
        new UnitValue(width, "in"),
        new UnitValue(height, "in"),
        300,
        ResampleMethod.BICUBIC
    );
}

// ------------------------------------------------------
// COLOR PROFILE CONVERSION
// ------------------------------------------------------
function convertToSRGB(doc) {
    doc.convertProfile("sRGB IEC61966-2.1", Intent.RELATIVECOLORIMETRIC, true, false);
}

// ------------------------------------------------------
// SAVE AS JPG
// ------------------------------------------------------
function saveAsJPG(doc, cropName, outputFolder) {
    var baseName = doc.name.replace(/\.[^\.]+$/, "");
    var outFile = new File(outputFolder + "/" + baseName + "_" + cropName + ".jpg");

    var opts = new JPEGSaveOptions();
    opts.quality = 12;
    opts.embedColorProfile = true;

    doc.saveAs(outFile, opts, true, Extension.LOWERCASE);
}

// ------------------------------------------------------
main();
